      PROGRAM VLEACH
*$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$
*
* Vadose Zone Leaching Model, Version 2.2c
* June 1998, Thomas Harter, University of California, Davis
*
* Based on the original VLEACH (version 1.0) developed by CH2M Hill
* for USEPA, Region IX
* This version updated by Varadhan Ravi (Dynamac Corp.) in March 1994
*
* This version (2.2) updated by Rashid Islam (CDSI) in October 1995
* Modified exponetial constant in the Millington equation for calculating 
* free-air diffusion coefficient from 10/3 to 7/3.  Also corrected a bug in 
* the Total ground water impact output.
*
* This version (2.2a) updated by Rashid Islam (CDSI) in June 1996
* (fixed the bugs in ground water impact calculations)
*
* This version (2.2b) updated by Thomas Harter (Dep. of Land, Air,
* and Water Resources, Univ. of California, Davis).  New input file
* format, new output file for use in Microsoft Excel(tm).
*
* This version (2.2c) recompiled by Thomas Harter (Dep. of Land, Air,
* and Water Resources, Univ. of California, Davis) for Windows XP. No
* changes in the program over Version 2.2b. Adjusted array dimensions
* to maximum of 100,000 cells and 500 printout times.
*
*$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$
*
* This program models liquid advection, gas diffusion, and
*    three-phase equilibration in the vadose zone.
* Current limitations include constant cell dimensions and
*    homogeneous soil properties throughout the profile.
* Advection solution is time-centered (Crank-Nicholson), space-upward.
* Diffusion solution is backward-difference (fully-implicit), space-centered.
* Simultaneous equations are solved in matrix form using the Thomas algorithm.

* Maximum number of cells and printout times is controlled by dimension
*    statements below.

      IMPLICIT REAL (M,K)

      COMMON /FILES/ IINP,IPRM,IOUT,IPRF,IPLT1,IPLT2,IGRF
      COMMON /CHEM/ KOC,KH,CMAX,DAIR
      COMMON /SIMUL/ DELT,STIME,PTIME,PRTIME,NTIME,DELZ,NCELL

* dimension on next line is max number of printout intervals (last one was
*    added on June 1996);
* (see also input check below, MAINS dimensions and format statements 1014, 1015)
      DIMENSION GTOTAL(500),GWIMP(500), agwimp(500)
* dimension on next line is max number of cells (see also input check in MAINS)
      DIMENSION CGAS(100000),CLIQ(100000),CS(100000),
     &   AGAS(3,100000),ALIQ(3,100000),RHS(100000)
      CHARACTER TITLE1*80

* dimensions of CLIQTIME are max number of cells by max number of printouts
	DIMENSION CLIQTIME(500,100000)

* Input data:
*  Simulation Parameters:
* TITLE1:       Description of problem
* NPOLY :       Number of Polygons
* DELT  :       Computational time-step (years)
* STIME :       Total simulation duration (years)
* PTIME :       Time interval for mass-balance & g.w. impact reports
* PRTIME:       Time interval for vertical concentration profile reports
*  Chemical Parameters:
* KOC   :       Organic carbon distribution coefficient (ml/g)
* KH    :       Henry's constant (dimensionless)
* CMAX  :       Aqueous solubility (mg/l)
* DAIR  :       Free air diffusion coefficient (m2/d)
*  Polygon-specific parameters:
* TITLE :       Description of polygon
* AREA  :       Area of polygon (sq. ft.)
* DELZ  :       Vertical spacing of cells (ft)
* Q     :       Groundwater recharge rate (ft/yr)
* RHOB  :       Dry bulk density of soil (g/ml)
* POR   :       Total effective porosity of soil (dimensionless)
* THETA :       Volumetric water content of soil (dimensionless)
* FOC   :       Organic carbon content of soil (dimensionless)
* CINF  :       Concentration of solute in infiltrating water (mg/l)
* CATM  :       Concentration of solute in atmosphere (
* CGW   :       Lower b'dry condition for gas diffusion:
*       CGW < 0., water table is impermeable to gas diffusion
*       CGW >= 0., fixed concentration at water table, (mg/l)
* NCELL :       Number of vertical cells


* prints heading and version number

      WRITE(*,77)
      WRITE(*,78)
	WRITE(*,79)
      WRITE(*,*)

* open files
      IINP=21
      IPRM=22
      IOUT=23
      IPRF=24
      IPLT1=25
      IPLT2=26
	IGRF=27
      OPEN (IINP,FILE='vleach-input.txt')
      OPEN (IPRM,FILE='VLEACH.PRM')
      OPEN (IOUT,FILE='VLEACH.OUT')
      OPEN (IPRF,FILE='VLEACH.PRF')
      OPEN (IPLT1,FILE='SOILIMP.DAT')
      OPEN (IPLT2,FILE='GWIMP.DAT')
      OPEN (IGRF,FILE='TRNSPROF.DAT')

* read overall simulation input data
      READ (IINP,14) 
      READ (IINP,11) TITLE1
      READ (IINP,12) NPOLY
      READ (IINP,13) DELT
      READ (IINP,13) STIME
      READ (IINP,13) PTIME
      READ (IINP,13) PRTIME
      READ (IINP,14) 
      READ (IINP,13) KOCI
      READ (IINP,13) KH
      READ (IINP,13) CMAXI
      READ (IINP,13) DAIRI


* ensure that number of output intervals does not exceed maximum dimensions

      TTTEMP=STIME/PRTIME
      IF (TTTEMP.GT.500) THEN
	  WRITE (*,*) "Increase the profile output time interval!"
	  WRITE (*,*) "(No more than 500 output times allowed)"
	  WRITE (*,*)
	  PAUSE
        STOP
	ENDIF
	  
11    FORMAT (40X,A)
12    FORMAT (40X,I3)
13    FORMAT (40X,G10.0)
14    FORMAT (40X)

* convert ml/g to ft3/g
      KOC = KOCI/28317.

* convert mg/l TO g/ft3
      CMAX = CMAXI*.028317
* convert m2/d TO ft2/yr
      DAIR = DAIRI*3929.

* write data back to IPRM
      WRITE(IPRM,77)
      WRITE(IPRM,78)
	WRITE(IPRM,79)
   77 FORMAT(' |------------------------------------------------------',
     2'------|'/
     3' |                VLEACH (Version 2.2c, 1998/2006)            |'/
     4' |                                                            |'/
     5' |                           By:                              |'/
     6' |        Thomas Harter, Univ. of California, Davis           |'/
     7' |                                                            |'/
     8' |                 based on the program by:                   |'/
     9' |        Varadhan Ravi and Jeffrey A. Johnson                |')
   78 FORMAT(' |                  (USEPA Contractors)                 '/
     2' |        Center for Subsurface Modeling Support              |'/
     3' |        Robert S. Kerr Environmental Research Laboratory    |'/
     4' |        U.S. Environmental Protection Agency                |'/
     5' |        P.O. Box 1198                                       |'/
     6' |        Ada, OK 74820                                       |')
   79 FORMAT(' |                                                      '/
     2' |        Based on the original VLEACH (version 1.0)          |'/
     3' |        developed by CH2M Hill, Redding, California         |'/
     4' |        for USEPA Region IX                                 |'/
     5' |------------------------------------------------------------|')
      WRITE(IPRM,*)
      WRITE (IPRM,16) TITLE1
      WRITE (IPRM,101) NPOLY
      WRITE (IPRM,102) DELT,STIME,PTIME,PRTIME
      WRITE (IPRM,103) KOCI,KOC
      WRITE (IPRM,104) KH
      WRITE (IPRM,105) CMAXI,CMAX
      WRITE (IPRM,106) DAIRI,DAIR

16	FORMAT(A)
101   FORMAT(I3,' polygons.')
102   FORMAT('Timestep = ',F6.2,' years.  Simulation length = ',
     &   F7.2,' years.'/'Printout every ',F6.2,' years.  ',
     &   'Vertical profile stored every ',F6.2,' years.')
103   FORMAT('Koc = ',G15.5,'ml/g,  ',G15.5,'cu.ft./g')
104   FORMAT('Kh = ',G15.5,' (dimensionless).')
105   FORMAT('Aqueous solubility =',G15.5,' mg/l, ',G15.5,' g/cu.ft')
106   FORMAT('Free air diffusion coefficient = ',G10.5,
     &   ' sq. m/day,  ',G10.5,' sq.ft./yr')


* set up output file
      WRITE(IOUT,77)
      WRITE(IOUT,78)
      WRITE(IOUT,*)
      WRITE (IOUT,11) TITLE1

      NTIME=INT(STIME/PTIME)

* initialize total gw impact array
      DO I=1,NTIME
          agwimp(i) = 0.   ! added on June 1996
          GTOTAL(I) = 0.
	ENDDO

* call MAIN for each polygon
      DO IPOLY=1,NPOLY

        WRITE (IGRF,*)
        WRITE(IPRM,107) IPOLY
107     FORMAT(//'Polygon ',I3)
        WRITE (IGRF,*)

        WRITE (IGRF,*)
        WRITE(IGRF,108) IPOLY
108     FORMAT('Liquid Phase Concentration [ug/l] for Polygon ',I3/
     &          '(cells by row, each column one time)'//)

      CALL MAINS(GWIMP,GTOTAL,CGAS,CLIQ,CS,AGAS,ALIQ,RHS,IPOLY,CLIQTIME)

c    Added on June 1996
* accumulate grand totals
        do isav = 1,ntime
           agwimp(isav) = agwimp(isav) + gwimp(isav)
        enddo

      ENDDO

* write grand total results to output file
      WRITE (IOUT,201)
      GCUM = 0.
      DO 250 IT = 1,NTIME
         GCUM = GCUM+GTOTAL(IT)
c        WRITE (IOUT,202) IT*PTIME,GWIMP(IT),GCUM
c      Modified on June 1996
         WRITE (IOUT,202) IT*PTIME,aGWIMP(IT),GCUM
250   CONTINUE

201   FORMAT(/'****************************************************'
     &   /'TOTAL GROUNDWATER IMPACT'//
     &   ' Time (yr)          Mass (g/yr)     Cumulative Mass (g)')
202   FORMAT(F10.2,5X,G15.5,3X,G15.5)

      STOP
      END


********************************************************************


      SUBROUTINE MAINS
     &    (GWIMP,GTOTAL,CGAS,CLIQ,CS,AGAS,ALIQ,RHS,IPOLY,CLIQTIME)

* this is the main subroutine which runs the simulation for each polygon.


      IMPLICIT REAL (M,K)

      COMMON /FILES/ IINP,IPRM,IOUT,IPRF,IPLT1,IPLT2,IGRF
      COMMON /CHEM/ KOC,KH,CMAX,DAIR
      COMMON /SIMUL/ DELT,STIME,PTIME,PRTIME,NTIME,DELZ,NCELL
      COMMON /BDRY/ CINF,CATM,CGW
      COMMON /SOIL/ RHOB,POR,THETA,FOC
      COMMON /PROCESS/ GASDIF,LIQADV,SORBED,GASPHS

      DIMENSION CGAS(*),CLIQ(*),CS(*),FACT(2,3),
     &   GWIMP(*),GTOTAL(*),AGAS(3,*),ALIQ(3,*),RHS(*),
     &   CLIQTIME(500,*)
	DIMENSION GRFTIME(500)
      LOGICAL LIQADV,GASDIF,SORBED,GASPHS
      CHARACTER TITLE*80,PLT*1

* read polygon-specific input data
      READ (IINP,15) 
      READ (IINP,11) TITLE
      READ (IINP,14) AREA
      READ (IINP,14) DELZ
      READ (IINP,14) Q
      READ (IINP,14) RHOBI
      READ (IINP,14) POR
      READ (IINP,14) THETA
      READ (IINP,14) FOC
      READ (IINP,14) CINFI
      READ (IINP,14) CATMI
      READ (IINP,14) CGWI
      READ (IINP,12) NCELL
      READ (IINP,13) PLT
      READ (IINP,14) PLTIME
	READ (IINP,15)

* ensure that problem does not exceed maximum dimensions

      IF (NCELL.GT.100000.) THEN
	  WRITE (*,*) "Reduce the number of cells to less than 100,001 !"
	  WRITE (*,*)
	  PAUSE
	  STOP
	ENDIF

* initialize variables

      DO I=1,NCELL
        CGAS(I)=0.
        CLIQ(I)=0.
        CS(I)=0.
	ENDDO

      DO IT=1,INT(STIME/PRTIME)+1
	  DO I=1,NCELL
	    CLIQTIME(IT,I)=0.
	  ENDDO
	ENDDO

11    FORMAT (40X,A)
12    FORMAT (40X,I6)
13    FORMAT (40X,A1)
14    FORMAT (40X,G10.0)
15    FORMAT (40X)
16    FORMAT (40X,2I10,G10.0)

* read initial conditions - for time being, assume total mass
60    READ (IINP,16) J1,J2,XCON
* convert input as ug/kg to g/ft**3
        xcon=xcon*rhobi*1e-6*28.31605
        IF (J2 .GT. NCELL) J2=NCELL
        DO ICELL = J1,J2
          CLIQ(ICELL)=XCON/(THETA)
	  ENDDO
      IF (J2 .LT. NCELL) GOTO 60

* convert g/ml to g/cu.ft.
      RHOB = RHOBI*28317.
* convery mg/l to g/cu.ft.
      CINF = CINFI*.028317
      CATM = CATMI*.028317
      CGW = CGWI*.028317
* write data back to IPRM
      WRITE(IPRM,11)TITLE
      WRITE(IPRM,201)AREA
      WRITE(IPRM,202)NCELL,DELZ
      WRITE(IPRM,203)RHOBI,RHOB,POR,THETA,FOC
      WRITE(IPRM,204)Q,CINFI,CINF,CATMI,CATM
      IF (CGW .LT. 0.) THEN
         WRITE(IPRM,205)
      ELSE
         WRITE(IPRM,206)CGWI,CGW
      ENDIF
	WRITE(IPRM,207)
	WRITE(IPRM,208)
	WRITE(IPRM,209)
     &  (I,THETA*CLIQ(I),(1.0E+06/28.31605)*CLIQ(I),I=1,NCELL)

201   FORMAT('Polygon area = ',G15.5,'sq. ft.')
202   FORMAT( I3,' cells, each cell ',F8.2,' ft. thick.')
203   FORMAT('Soil Properties:'/' Bulk density =',F8.3,
     &  'g/ml,  ',F12.3,'g/cu.ft.'/' Porosity = ',F8.3,
     &  ' Volumetric water content = ',F8.4/
     &  ' Organic carbon content = ',F8.6)
204   FORMAT('Recharge Rate = ',F10.4,' ft/yr'/
     &'Conc. in recharge water = ',G10.5,'m g/l,  ',G10.5,' g/cu.ft'/
     &'Atmospheric concentration = ',F18.8,' mg/l,  ',F18.8,' g/cu.ft')
205   FORMAT('Water table is impermeable to gas diffusion.')
206   FORMAT('Water table has a fixed concentration of ',G10.5,
     & 'mg/l,  ',G10.5,'g/cu.ft.'/'   with respect to gas diffusion.')
207   FORMAT('Initial Mass and Concentration:')
208   FORMAT
     & ('  cell',' - total mass [g/cu.ft]',' - equiv.c_liq[ug/l]')
209   FORMAT(I6,G20.5,G20.5)

* check for active processes
      GASPHS = .TRUE.
      IF (KH .EQ. 0.) GASPHS = .FALSE.

      GASDIF = .TRUE.
      IF (KH .EQ. 0. .OR. DAIR .EQ. 0.) GASDIF = .FALSE.

      LIQADV = .TRUE.
      IF (Q .EQ. 0.) LIQADV = .FALSE.

      SORBED = .TRUE.
      IF (KOC .EQ. 0. .OR. FOC .EQ. 0.) SORBED = .FALSE.

* initial calculations - gas diffusion
      IF (GASDIF) THEN
* Modified this equation from 10/3 to 7/3 ( October 1995)
         D = DAIR*((POR-THETA)**(7./3.))/(POR*POR)
         ALPHA = (DELT*D)/(DELZ*DELZ)
         CALL IGAS(ALPHA,AGAS)
      ENDIF

* initial calculations - liquid advection
      IF (LIQADV) THEN
        BETA = (Q*DELT)/(2.*THETA*DELZ)
        CALL ILIQ(BETA,ALIQ)
      ENDIF

* initial calculations - equilibration
      CALL IEQUIL(FACT)


* initial equilibration
      CALL EQUIL(CGAS,CLIQ,CS,MGTOT,MLTOT,MSTOT,MTOTAL,FACT,IFLAG)
      IF (IFLAG .GT. 0) WRITE (IOUT,401) 0.,IFLAG

401   FORMAT('WARNING!!!  At time = ',F10.2,
     &   ', aqueous solubility was exceeded in ',I3,' cells.')

      WRITE (IOUT,301) IPOLY,0.,MTOTAL,MGTOT,MLTOT,MSTOT
301   FORMAT(//'Polygon ',I3/'At time = ',F10.2,
     &   ', total mass in vadose zone =',G15.5,'g/sq.ft.'/
     &   'Mass in gas phase     =  ',G15.5,'g/sq.ft.'/
     &   'Mass in liquid phase  =  ',G15.5,'g/sq.ft.'/
     &   'Mass sorbed           =  ',G15.5,'g/sq.ft.')

      MT0 = MTOTAL
      MTP = MTOTAL
      MLTCUM = 0.
      MLBCUM = 0.
      MGTCUM = 0.
      MGBCUM = 0.
      MLTINT = 0.
      MLBINT = 0.
      MGTINT = 0.
      MGBINT = 0.

* initialize for time steps
      ITIME = 0
	IPRTIME = 1

      WRITE (*,1001) IPOLY
1001  FORMAT ('+Beginning Calculations for Polygon ',I3)

* write vertical concentration profiles
      WRITE(IPRF,77)
      WRITE(IPRF,78)
   77 FORMAT(' |------------------------------------------------------',
     2'------|'/
     3' |                VLEACH (Version 2.2a, 1996)                 |'/
     4' |                                                            |'/
     5' |                           By:                              |'/
     6' |        Varadhan Ravi and Jeffrey A. Johnson                |'/
     7' |                  (USEPA Contractors)                       |'/
     8' |        Center for Subsurface Modeling Support              |'/
     9' |        Robert S. Kerr Environmental Research Laboratory    |')
   78 FORMAT(' |'8X'U.S. Environmental Protection Agency'16X'|'/
     2' |        P.O. Box 1198                                       |'/
     3' |        Ada, OK 74820                                       |'/
     4' |                                                            |'/
     5' |        Based on the original VLEACH (version 1.0)          |'/
     6' |        developed by CH2M Hill, Redding, California         |'/
     7' |        for USEPA Region IX                                 |'/
     8' |------------------------------------------------------------|')
      WRITE(IPRF,*)
      WRITE(IPRF,101) TITLE,0.
      DO I=1,NCELL
	  CLIQTIME(IPRTIME,I) = CLIQ(I)/0.000028317
        WRITE(IPRF,102) I,CGAS(I),CLIQ(I),CS(I)
      ENDDO

1000  ITIME = ITIME + 1
      TIME = ITIME * DELT

* gas diffusion step
      IF (GASDIF) THEN

        CALL GAS(AGAS,CGAS,ALPHA,RHS)

* b'dry flux calculations
* fully implicit
        MGT = (POR-THETA)*DELT*D*(CATM - CGAS(1))/DELZ
        MGB = (POR-THETA)*DELT*D*(CGW - CGAS(NCELL))/DELZ
        IF (CGW .LT. 0.) MGB = 0.
        IF (CATM .LT. 0.) MGT = 0.
      ENDIF

* liquid advection step
      IF (LIQADV) THEN

        CBOT = CLIQ(NCELL)

        CALL LIQ(ALIQ,CLIQ,BETA,RHS)

* b'dry flux calculations
        MLT = DELT*Q*CINF
        MLB = -DELT*Q*(CBOT+CLIQ(NCELL))/2.

      ENDIF

* mass equilibration step
      CALL EQUIL(CGAS,CLIQ,CS,MGTOT,MLTOT,MSTOT,MTOTAL,FACT,IFLAG)
      IF (IFLAG .GT. 0) WRITE (IOUT,401) TIME,IFLAG

* mass balance calculations
      MLTCUM = MLTCUM + MLT
      MLBCUM = MLBCUM + MLB
      MGTCUM = MGTCUM + MGT
      MGBCUM = MGBCUM + MGB
      MLTINT = MLTINT + MLT
      MLBINT = MLBINT + MLB
      MGTINT = MGTINT + MGT
      MGBINT = MGBINT + MGB

*  Write groundwater impact data to plot file
      IF(PLT.EQ.'Y' .OR. PLT.EQ.'y')THEN
        WRITE(IPLT2,*)TIME,-AREA*(MLB+MGB)/DELT
      END IF

*  Write soil concentration - depth data to plot file
      IF((PLT.EQ.'Y' .OR. PLT.EQ.'y') .AND. (TIME.EQ.PLTIME))THEN
        DO I=1,NCELL
          WRITE(IPLT1,*)CS(I),-(I-1./2.)*DELZ
        ENDDO
      END IF

* Harter (6/9/98):
* Do even if MOD function does not come out exactly 0 due
* to roundoff error (problem w/ MS Fortran4 compiler for
* timesteps less than 1 year).  This works for time steps
* of 0.0001 and larger.
      IF (ABS((TIME/PTIME)-ANINT(TIME/PTIME)).LT.0.0001) THEN
*        IF (MOD(TIME,PTIME) .EQ. 0.) THEN

c       Modification! done by Rashid Islam (10/02/93)  
c       A bug was found in the grand total mass calculation
c       Last modification (6/3/96) 
*       Harter (6/10/98): change INT to NINT (roundoff rather than truncate)
      
        GWIMP(NINT(TIME/PTIME)) = -AREA*(MLB+MGB)/DELT
c       GTOTAL(INT(TIME/PTIME))= -AREA*(MLBINT+MGBINT)
      
        GTOTAL(NINT(TIME/PTIME))= gtotal(nint(time/ptime))  
     &      -AREA*(MLBINT+MGBINT)
        WRITE (*,1002) IPOLY,TIME
1002    FORMAT ('+Calculating Polygon ',I3,' at time ',F10.4)

* write output data

        CALL OUTPUT(TIME,MT0,MTP,MLTCUM,MLBCUM,MGTCUM,MGBCUM,
     &   MLTINT,MLBINT,MGTINT,MGBINT,MGTOT,MLTOT,MSTOT,MTOTAL,IPOLY)

      ENDIF


* Harter (6/9/98):
* Do even if MOD function does not come out exactly 0 due
* to roundoff error (problem w/ MS Fortran4 compiler for
* timesteps less than 1 year).  This works for time steps
* of 0.0001 and larger.

* record profile in array CLIQTIME for later printout

      IF (ABS((TIME/PRTIME)-ANINT(TIME/PRTIME)).LT.0.0001) THEN

	  IPRTIME=IPRTIME+1
        WRITE(IPRF,101) TITLE,TIME
	  GRFTIME(IPRTIME)=TIME
        DO I=1,NCELL
	    CLIQTIME(IPRTIME,I) = CLIQ(I)/0.000028317
          WRITE(IPRF,102) I,CGAS(I),CLIQ(I),CS(I)
        ENDDO

      ENDIF


101   FORMAT (/A/'Time: 'F10.3/
     &   'Cell     Cgas(g/cu.ft)      Cliq(g/cu.ft)      Csol(g/g)')
102   FORMAT (I5,3G17.5)


      IF (TIME .LT. STIME) GOTO 1000

      WRITE (*,1003) IPOLY
1003  FORMAT ('+Polygon ',I3,' complete.                        ')

      WRITE (IOUT,1011) IPOLY
      DO IT = 1,NTIME
        WRITE (IOUT,1012) IT*PTIME,GWIMP(IT)/AREA,GWIMP(IT)
      ENDDO
      WRITE (IOUT,1013)

1011  FORMAT(//'GROUNDWATER IMPACT OF POLYGON ',I3//
     &   ' Time        Mass flux (g/yr/sq.ft.)      Total Mass(g/yr)')
1012  FORMAT(F10.2,5X,G15.5,11X,G15.5)
1013  FORMAT('****************************************************')

* Harter (6/9/98):
* write out profiles of all times (array CLIQTIME) into IGRF

	WRITE (IGRF,1014) NCELL,(GRFTIME(IT),IT=1,IPRTIME)
      DO I = 1,NCELL
	  DEPTH = (I-0.5)* DELZ 
        WRITE (IGRF,1015) DEPTH,(CLIQTIME(IT,I),IT=1,IPRTIME)
     	ENDDO
1014	FORMAT(I6,500G15.5)
1015  FORMAT(F20.6,500G15.5)

      RETURN
      END

*******************************************************************

      SUBROUTINE IGAS(ALPHA,AGAS)

* this subroutine sets up the left-hand side matrix for gas
* diffusion, and reduces it using the Thomas algorithm

      IMPLICIT REAL (M,K)

      COMMON /SIMUL/ DELT,STIME,PTIME,PRTIME,NTIME,DELZ,NCELL
      COMMON /BDRY/ CINF,CATM,CGW
      DIMENSION AGAS(3,NCELL)

      DO I=1,3
        DO J=1,NCELL
          AGAS(I,J)=0.
        ENDDO
	ENDDO

      A1 = -ALPHA
      A2 = 1.+2.*ALPHA

      DO I=1,NCELL
        AGAS(1,I) = A1
        AGAS(2,I) = A2
        AGAS(3,I) = A1
      ENDDO

* impermeable lower boundary
      IF (CGW .LT. 0.) AGAS(2,NCELL) = 1.+ALPHA
* impermeable upper boundary
      IF (CATM .LT. 0.) AGAS(2,1) = 1.+ALPHA

      CALL THOMAS(AGAS,NCELL)

      RETURN
      END


*****************************************************************

      SUBROUTINE ILIQ(BETA,ALIQ)

* this subroutine sets up the left-hand side matrix for liquid
* advection, and reduces it using the Thomas algorithm

      IMPLICIT REAL (M,K)

      COMMON /SIMUL/ DELT,STIME,PTIME,PRTIME,NTIME,DELZ,NCELL
      DIMENSION ALIQ(3,NCELL)

      DO I=1,3
        DO J=1,NCELL
          ALIQ(I,J)=0.
        ENDDO
	ENDDO

      A1 = -BETA
      A2 = 1.+BETA

      DO I=1,NCELL
        ALIQ(1,I) = A1
        ALIQ(2,I) = A2
	ENDDO

      CALL THOMAS(ALIQ,NCELL)

      RETURN
      END

*************************************************************

      SUBROUTINE IEQUIL(FACT)

      IMPLICIT REAL (M,K)

      COMMON /SOIL/ RHOB,POR,THETA,FOC
      COMMON /SIMUL/ DELT,STIME,PTIME,PRTIME,NTIME,DELZ,NCELL
      COMMON /CHEM/ KOC,KH,CMAX,DAIR
      COMMON /PROCESS/ GASDIF,LIQADV,SORBED,GASPHS
      COMMON /BDRY/ CINF,CATM,CGW

      LOGICAL GASDIF,LIQADV,SORBED,GASPHS
      DIMENSION FACT(2,3)

* initial calculations - equilibrium factors
      KD = KOC*FOC
      CGW = CGW*KH
* FACT(1,I) converts from concentration in phase I to mass in phase I
* phases: 1:gas, 2:liquid, 3: sorbed
      FACT(1,1) = (POR-THETA)*DELZ
      FACT(1,2) = THETA*DELZ
      FACT(1,3) = RHOB*DELZ
* FACT(2,I) partitions total mass to concentration in phase I
      IF (GASPHS) THEN
        FACT(2,1) = 1./(DELZ*(THETA/KH + (POR-THETA) + RHOB*KD/KH))
      ELSE
        FACT(2,1) = 0.
      ENDIF

      FACT(2,2) = 1./(DELZ*(THETA + (POR-THETA)*KH + RHOB*KD))

      IF (SORBED) THEN
        FACT(2,3) = 1./(DELZ*(THETA/KD + (POR-THETA)*KH/KD + RHOB))
      ELSE
        FACT(2,3) = 0.
      ENDIF

      RETURN
      END

**************************************************************

      SUBROUTINE GAS(AGAS,CGAS,ALPHA,RHS)
* this subroutine sets up the right-hand side of the gas-diffusion
* calculation, calls SOLVET to do the calculation, computes b'dry fluxes
* time-forward, fully implicit

      IMPLICIT REAL (M,K)

      COMMON /SIMUL/ DELT,STIME,PTIME,PRTIME,NTIME,DELZ,NCELL
      COMMON /BDRY/ CINF,CATM,CGW
      DIMENSION AGAS(3,NCELL),CGAS(NCELL),RHS(NCELL)

      DO I=1,NCELL
        RHS(I) = CGAS(I)
      ENDDO

* fixed lower b'dry
      IF (CGW .GE. 0.) RHS(NCELL) = CGAS(NCELL)+ALPHA*CGW

* fixed upper b'dry
      IF (CATM .GE. 0.) RHS(1) = CGAS(1)+ALPHA*CATM

      CALL SOLVET(AGAS,CGAS,RHS,NCELL)

      RETURN
      END

*************************************************************

      SUBROUTINE LIQ(ALIQ,CLIQ,BETA,RHS)
* this subroutine sets up the right-hand side of the liquid advection
* calculation, calls SOLVET to do the calculation, computes b'dry fluxes

      IMPLICIT REAL (M,K)

      COMMON /SIMUL/ DELT,STIME,PTIME,PRTIME,NTIME,DELZ,NCELL
      COMMON /BDRY/ CINF,CATM,CGW
      DIMENSION ALIQ(3,NCELL),CLIQ(NCELL),RHS(NCELL)

      RHS(1) = CLIQ(1)-BETA*(CLIQ(1)-2.*CINF)

      DO I=2,NCELL
        RHS(I) = CLIQ(I)-BETA*(CLIQ(I)-CLIQ(I-1))
      ENDDO

      CALL SOLVET(ALIQ,CLIQ,RHS,NCELL)


      RETURN
      END

*************************************************************

      SUBROUTINE EQUIL(CGAS,CLIQ,CS,MGTOT,MLTOT,MSTOT,
     &    MTOTAL,FACT,IFLAG)

* this subroutine re-equilibrates all three phases

      IMPLICIT REAL (K,M)

      COMMON /SIMUL/ DELT,STIME,PTIME,PRTIME,NTIME,DELZ,NCELL
      COMMON /CHEM/ KOC,KH,CMAX,DAIR
      DIMENSION CGAS(NCELL),CLIQ(NCELL),CS(NCELL),FACT(2,3)

      MGTOT = 0.
      MLTOT = 0.
      MSTOT = 0.
      MTOTAL = 0.
      IFLAG = 0

      DO N=1,NCELL

        MT = CGAS(N)*FACT(1,1) + CLIQ(N)*FACT(1,2) + CS(N)*FACT(1,3)
        MTOTAL = MTOTAL + MT

        CGAS(N) = MT * FACT(2,1)
        CLIQ(N) = MT * FACT(2,2)
        CS(N) = MT * FACT(2,3)

        MG = CGAS(N)*FACT(1,1)
        ML = CLIQ(N)*FACT(1,2)
        MS = CS(N)*FACT(1,3)
        MGTOT = MGTOT + MG
        MLTOT = MLTOT + ML
        MSTOT = MSTOT + MS

        IF (CLIQ(N) .GT. CMAX) IFLAG=IFLAG+1

      ENDDO

      RETURN
      END

************************************************************************

      SUBROUTINE OUTPUT(TIME,MT0,MTP,MLTCUM,MLBCUM,MGTCUM,MGBCUM,
     &   MLTINT,MLBINT,MGTINT,MGBINT,MGTOT,MLTOT,MSTOT,MTOTAL,IPOLY)

      IMPLICIT REAL (M,K)

      COMMON /FILES/ IINP,IPRM,IOUT,IPRF,IPLT1,IPLT2,IGRF
      COMMON /SIMUL/ DELT,STIME,PTIME,PRTIME,NTIME,DELZ,NCELL

      WRITE (IOUT,201) IPOLY,TIME,MTOTAL,MGTOT,MLTOT,MSTOT
201   FORMAT(//'Polygon ',I3/'At time = ',F12.4,
     &   ', total mass in vadose zone =',G15.5,'g/sq.ft.'/
     &   'Mass in gas phase     =  ',G15.5,'g/sq.ft.'/
     &   'Mass in liquid phase  =  ',G15.5,'g/sq.ft.'/
     &   'Mass sorbed           =  ',G15.5,'g/sq.ft.')

      DEL1 = MTOTAL-MTP
      DEL2 = MLTINT+MLBINT+MGTINT+MGBINT
      WRITE (IOUT,102) TIME-PTIME
      WRITE (IOUT,103) DEL1
      WRITE (IOUT,104) MLTINT,MLBINT,MGTINT,MGBINT
      WRITE (IOUT,105) DEL2
      WRITE (IOUT,106) DEL1-DEL2

      DEL1 = MTOTAL-MT0
      DEL2 = MLTCUM+MLBCUM+MGTCUM+MGBCUM
      WRITE (IOUT,107)
      WRITE (IOUT,103) DEL1
      WRITE (IOUT,104) MLTCUM,MLBCUM,MGTCUM,MGBCUM
      WRITE (IOUT,105) DEL2
      WRITE (IOUT,106) DEL1-DEL2

102   FORMAT(/'Since last printout at time = ',F12.4)
103   FORMAT(15X,'Change in Total Mass = ',G15.5,'g/sq.ft.')
104   FORMAT(20X,'Advection in from atmosphere =  ',G15.5,'g/sq.ft.'/
     &         20X,'Advection in from water table = ',G15.5,'g/sq.ft.'/
     &         20X,'Diffusion in from atmosphere =  ',G15.5,'g/sq.ft.'/
     &         20X,'Diffusion in from water table = ',G15.5,'g/sq.ft.')
105   FORMAT(15X,'Total inflow at boundaries = ',G15.5,'g/sq.ft.')
106   FORMAT(15X,'Mass discrepancy = ',G15.5,'g/sq.ft.')
107   FORMAT(/'Since beginning of run at time = 0.0')


      MLTINT = 0.
      MLBINT = 0.
      MGTINT = 0.
      MGBINT = 0.
      MTP = MTOTAL

      RETURN
      END


      SUBROUTINE THOMAS (A,N)

*       this subroutine solves an asymmetric tridiagonal matrix
*       by Thomas' algorithm, a special case of Crout's method.
*       See S.P. Neuman's lecture notes for Hydr 504.

      DIMENSION A(3,N)

*       see Shlomo's notes for nomenclature
*       matrix A:   on input        on output
*         A[1,]        a               a
*         A[2,]        b               m
*         A[3,]        c               u


*       produce u and m vectors
      DO I=1,N-1
        A(3,I)=A(3,I)/A(2,I)
        A(2,I+1)=A(2,I+1)-A(1,I+1)*A(3,I)
      ENDDO

      RETURN
      END



********************************************************************



      SUBROUTINE SOLVET (A,X,D,N)

*     this subroutine solves the system Ax = d, once matrix
*     A has been decomposed by subroutine THOMAS

      DIMENSION A(3,N),X(N),D(N)


*     forward substitute to find Y,and store it in D
      
      D(1)=D(1)/A(2,1)
      DO I=2,N
        D(I)=(D(I)-A(1,I)*D(I-1))/A(2,I)
	ENDDO


*     back substitute to find X
      X(N)=D(N)
      DO I=N-1,1,-1
        X(I)=D(I)-A(3,I)*X(I+1)
      ENDDO


      RETURN
      END
